/*
 * Indie Game Jame 2 - example game "framework"
 * 2004/3/15 checker@d6.com
 */
#include "shared/core/first.h"

#pragma warning(disable:4244)

// windows crap
#define WIN32_LEAN_AND_MEAN
#define WIN32_EXTRA_LEAN
#include <windows.h>
#include <windowsx.h>

// system
#include <assert.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <time.h>

// gl stuff
#include <gl/gl.h>
#include "glutils.h"

// physics stuff
#include "shared/physics/2d/2d_body_hull.h"
#include "shared/physics/2d/2d_body_list.h"
#include "shared/physics/2d/2d_body_box.h"
#include "shared/physics/2d/2d_body_sphere.h"
#include "shared/physics/2d/rigid/2d_rigid_sim.h"
#include "shared/physics/2d/rigid/2d_rigid_drag_force.h"
#include "shared/physics/2d/rigid/2d_rigid_gravity_force.h"
#include "shared/physics/2d/rigid/2d_rigid_spring_force.h"
#include "shared/physics/2d/rigid/2d_rigid_control_force.h"
#include "../../utils/misc/framerate.h"
#include "shared/input/joystick.h"
using namespace ajb;
using namespace Physics::D2;
using namespace Physics::D2::Rigid;

// misc
#include "utils.h"
#include <vector>
using namespace std;

// media lib
#include "music.h"
#include "zsound.h"
#include "bitmap.h"

#include "property_system.h"
#include "property_system_editor.h"
#include "editor.h"

/********************************* globals **********************************/

// window/camera stuff
char const AppName[] = "IGJ2 - PigPlow - by Chris Carollo";
int WindowWidth = 1024;
int WindowHeight = 768;
bool AppQuit = false;  // stuff this to kill yourself
float FOVY = 45.0f, ZNear = 0.1f, ZFar = 1000.0f;  // whatever, we're 2D
float AspectRatio = float(WindowWidth)/float(WindowHeight); // updated in Resize
float CameraAngleRadians = 0.0, CameraX = 0.0, CameraY = 0.0, CameraZ = 700.0;
Vec2 WorldFromWindowCoords( Vec2 screen );
Vec2 WindowFromWorldCoords( Vec2 world );

// win32 stuff
HWND WindowHandle;
HDC WindowDC;

// gl stuff
HGLRC WindowRC;

// functions for you to play in
void Resize( int w, int h );
void Initialize( void );
void Uninitialize( void );
void Idle( void );
LONG FAR PASCAL AppWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

GLuint iconTextureShot;
GLuint iconTextureExplosion;
GLuint iconTextureAttractor;
GLuint iconTextureUp;
GLuint iconTextureDown;

GLuint dozerTexture[4];

GLuint backgroundTexture;
GLuint pigTexture[16];

GLuint titleTexture;

// system
Music music;

// physics stuff
Sim simulator;
Drag_Force    global_drag(.99f, .99f);
Gravity_Force gravity(Vec2(0.0f, -1.0f));
Rigid_Body    world_body(make_Body_Null(),Vec2(0,0),0,0,0);

#define kSimFramerate  (60)

// avg seconds between respawns
static int iconSpawnSeconds[5] = { 4, 4, 4, 15, 15 };

#define kGameDuration  (10.0f)

static float GameTimer = kGameDuration;

static float fadeAlpha = 0.0f;

PROP_PARSE_THIS_FILE;
PROP_STRUCT struct global_properties
{
    float Gravity;
    
    float MouseSpringKs;
    float MouseSpringKd;
    float MouseSpringRestLength;

	float WinnerX;
	float WinnerY;

	float IsWinnerX;
	float IsWinnerY;

	float TopScoreX;
	float TopScoreY;

	float TopScoreValX;
	float TopScoreValY;

	float BestStreakX;
	float BestStreakY;

	float BestStreakValX;
	float BestStreakValY;

	float AnyKeyX;
	float AnyKeyY;
};

static global_properties Globals;

struct game_object;

// Collision bits
// 0-4 players


enum eObjType
{
	kFrame         =  0,
	kPig           =  1,
	kGoal1         =  2,
	kGoal2         =  3,
	kGoal3         =  4,
	kGoal4         =  5,
	kPlayer1       =  6,
	kPlayer2       =  7,
	kPlayer3       =  8,
	kPlayer4       =  9,
	kShot          = 10,
	kIconShot      = 20,
	kIconExplosion = 21,
	kIconAttractor = 22,
	kIconUp        = 23,
	kIconDown      = 24,
	kPlayerPhantom = 25
};

int randRange(int min, int max)
{
	return (rand() % (max - min + 1)) + min;
}

bool IsPlayer(eObjType objType)
{
	return ((objType >= kPlayer1) && (objType <= kPlayer4));
}

int GetPlayer(eObjType objType)
{
	return ((int)objType - (int)kPlayer1);
}

bool IsGoal(eObjType objType)
{
	return ((objType >= kGoal1) && (objType <= kGoal4));
}

int GetGoal(eObjType objType)
{
	return ((int)objType - (int)kGoal1);
}

bool IsPowerup(eObjType objType)
{
	return ((objType >= kIconShot) && (objType <= kIconDown));
}

int GetPowerup(eObjType objType)
{
	return ((int)objType - (int)kIconShot);
}

void SetCollisionMask(Rigid_Body* pRB, u32 mask);
void PlaySqueal(int which, float pan = 1.0f);

// game stuff
struct game_object {
    
    game_object( char *texture_file , Rigid_Body *rbInit, eObjType _objType ) :
            rb(rbInit), objType(_objType), bActive(true), playerAttached(-1), pSpring(0), lastAnimLoc(0,0), animFrame(0)
        {
            assert(rb);

			if (objType == kPig)
			{
				animFrame = randRange(1, 15);
			}
			if (objType == kPlayerPhantom)
			{
				SetCollisionMask(rb, 0xF);
			}
			else
			{
				u32 mask = ~0xF;
				if (IsGoal(objType))
				{
					mask &= ~0x10;
				}
				if (IsPlayer(objType))
				{
					mask = 0x10;
				}
				SetCollisionMask(rb, mask);
			}
			rb->m_pAppData = this;
            simulator.add(*rb);
        }

    ~game_object( void )
        {
            delete rb;
        }
    
	int playerAttached;
	Spring_Force* pSpring;

	Vec2 lastAnimLoc;
	int  animFrame;

	bool bActive;
	eObjType objType;
    Rigid_Body *rb;
};

void SetCollisionMask(Rigid_Body* pRB, u32 mask)
{
	if (pRB->m_Pbody->type() == Body::LIST)
	{
		Body_List& bl = my_static_cast<Body_List&>(*(pRB->m_Pbody));
		for (int i=0; i<bl.num_bodies(); i++)
		{
			bl.body(i).m_collision_plane = mask;
		}
	}
	else
	{
		pRB->m_Pbody->m_collision_plane = mask;
	}
}

static vector<game_object*> frames;
static vector<game_object*> pigs;
static vector<game_object*> players;
static vector<game_object*> goals;
static vector<game_object*> icons;
static vector<game_object*> shots;
static vector<game_object*> attachedshots[4];
static vector<game_object*> playerphantoms;

const int kDefaultCollisionMaskPig  = 0xFFFFFFFF;
const int kDefaultCollisionMaskShot = 0xFFFFFFFF;

static float GameDoneTimer;
static bool bGameDoneTimerSet = false;

static int playerScore[4];
static int playerStreak;
static int lastPlayerScored = -1;

static int bestStreakVal;
static int bestStreakPlayer;

static vector<Constant_Force*> player_thrust;
static vector<Angle_Spring_Force*> player_torque;

static vector<game_object*> objs;
static vector<Binary_Force *> forces;

static Vec2 world_mouse, screen_mouse;
static Joystick PS2Controllers;

static Binary_Force *spring = 0;

GLuint LoadTexture(char *Filename)
{
    GLuint texture;
    
    Bitmap_Load_Result result;
    verify(LoadBitmapByExtension(Filename,&result));

    // gl stuff
    texture = gl_texture_from_bitmap(result.data, result.width, result.height,
                                     result.bytes_per_pixel);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);

    // delete the bitmap
    delete [] result.data;

    return(texture);
}

void glBody1v(Body *body, bool Solid)
{
    if(body)
    {
		{
			Body_Hull *p = dynamic_cast<Body_Hull*>(body);
			if(p)
			{
				glBegin(Solid ? GL_TRIANGLE_FAN : GL_LINE_LOOP);
				for(int i = 0;i<p->num_verts();++i)
				{
					Vec2 worldspace = p->vert(i);
					glVertex2f(worldspace.x(),worldspace.y());
				}
				glEnd();
			}
		}

		{
			Body_Box *p = dynamic_cast<Body_Box*>(body);
			if(p)
			{
				glBegin(Solid ? GL_TRIANGLE_FAN : GL_LINE_LOOP);
				for(int i = 0;i<p->num_verts();++i)
				{
					Vec2 worldspace = p->vert(i);
					glVertex2f(worldspace.x(),worldspace.y());
				}
				glEnd();
			}
		}

		{
			Body_Sphere *p = dynamic_cast<Body_Sphere*>(body);
			if(p)
			{
				glBegin(Solid ? GL_TRIANGLE_FAN : GL_LINE_LOOP);
				int total_steps = 16;
				Vec2 center = p->x();
				if(Solid)
				{
					glVertex2f(center.x(), center.y());
				}
				for(int a = 0;a<total_steps;++a)
				{
					float Angle = 2.0f * 3.14159265358979323846f * ((float)a / (float)(total_steps - 1));
					float Radius = p->r();
					Vec2 arm(Radius*cos(Angle), Radius*sin(Angle));
					arm += center;
					glVertex2f(arm.x(), arm.y());
				}
				glEnd();
			}
		}

        Body_List *l = dynamic_cast<Body_List*>(body);
        if(l)
        {
            {for(int Index = 0;
                 Index < l->num_bodies();
                 ++Index)
            {
                glBody1v(&l->body(Index), Solid);
            }}
        }
    }
}

void ClearMouseSpring(void)
{
    if(spring)
    {
        simulator.remove(*spring);
        delete spring;
        spring = 0;
    }
}

bool HasPlayerAttached(game_object* pGameObj)
{
	return (pGameObj->playerAttached >= 0);
}

void DetachFromPlayer(game_object* pGameObj)
{
	if (!HasPlayerAttached(pGameObj))  return;

	switch (pGameObj->objType)
	{
		case kShot: 
		{
			attachedshots[pGameObj->playerAttached].erase(std::find(attachedshots[pGameObj->playerAttached].begin(), attachedshots[pGameObj->playerAttached].end(), pGameObj));  
			SetCollisionMask(pGameObj->rb, kDefaultCollisionMaskShot);
			break;
		}
	}

	simulator.remove(*(pGameObj->pSpring));
	
	delete pGameObj->pSpring;
	pGameObj->pSpring = 0;
	pGameObj->playerAttached = -1;
}

void AttachToPlayer(game_object* pGameObj, int playerIndex, float ks, float kd)
{
	if (HasPlayerAttached(pGameObj))  DetachFromPlayer(pGameObj);

	Handle hPlayer(*(players[playerIndex]->rb), Vec2(0,0));
	Handle hAttachment(*(pGameObj->rb), Vec2(0,0));

	pGameObj->playerAttached = playerIndex;
	pGameObj->pSpring = new Spring_Force(hPlayer, hAttachment, ks, kd, 0.1f);
	simulator.add(*(pGameObj->pSpring));

	switch (pGameObj->objType)
	{
		case kShot: 
		{
			attachedshots[playerIndex].push_back(pGameObj);  
			SetCollisionMask(pGameObj->rb, kDefaultCollisionMaskShot & ~(1 << playerIndex));
			break;
		}
	}
}

#define kEdgeImpulse  (100.0f)

void handleCollisions()
{
	for (Contact_List::const_iterator cit=simulator.m_contacts.begin(); cit != simulator.m_contacts.end(); cit++)
	{
		game_object* pGameObj1 = (game_object*)((Rigid_Body *)(cit->m_Prb_a))->m_pAppData;
		game_object* pGameObj2 = (game_object*)((Rigid_Body *)(cit->m_Prb_b))->m_pAppData;

		game_object* pGoalObj = 0;
		game_object* pPigObj = 0;
		game_object* pPlayerObj = 0;
		game_object* pPowerupObj = 0;
		game_object* pFrameObj = 0;

		if (pGameObj1 && pGameObj2)
		{
			// Pig vs Goal
			if (IsGoal(pGameObj1->objType) && (pGameObj2->objType == kPig))
			{
				pGoalObj = pGameObj1;
				pPigObj = pGameObj2;
			}
			if (IsGoal(pGameObj2->objType) && (pGameObj1->objType == kPig))
			{
				pGoalObj = pGameObj2;
				pPigObj = pGameObj1;
			}

			// Pig vs Frame
			if ((pGameObj1->objType == kPig) && (pGameObj2->objType == kFrame))
			{
				pPigObj = pGameObj1;
				pFrameObj = pGameObj2;
			}
			if ((pGameObj2->objType == kPig) && (pGameObj1->objType == kFrame))
			{
				pPigObj = pGameObj2;
				pFrameObj = pGameObj1;
			}

			// Pig vs Player
			if (IsPlayer(pGameObj1->objType) && (pGameObj2->objType == kPig))
			{
				pPlayerObj = pGameObj1;
				pPigObj = pGameObj2;
			}
			if (IsPlayer(pGameObj2->objType) && (pGameObj1->objType == kPig))
			{
				pPlayerObj = pGameObj2;
				pPigObj = pGameObj1;
			}

			// Player vs powerup
			if (IsPlayer(pGameObj1->objType) && IsPowerup(pGameObj2->objType))
			{
				pPlayerObj = pGameObj1;
				pPowerupObj = pGameObj2;
			}
			if (IsPlayer(pGameObj2->objType) && IsPowerup(pGameObj1->objType))
			{
				pPlayerObj = pGameObj2;
				pPowerupObj = pGameObj1;
			}
		}

		// Handle frame collisions
		if (pFrameObj && pPigObj && pPigObj->bActive)
		{
			if (pPigObj->rb->x.x() < -300) pPigObj->rb->v += Vec2(kEdgeImpulse, 0);
			if (pPigObj->rb->x.x() >  300) pPigObj->rb->v -= Vec2(kEdgeImpulse, 0);
			if (pPigObj->rb->x.y() < -200) pPigObj->rb->v += Vec2(0, kEdgeImpulse);
			if (pPigObj->rb->x.y() >  200) pPigObj->rb->v -= Vec2(0, kEdgeImpulse);
		}

		// Handle powerups
		if (pPlayerObj && pPowerupObj && pPowerupObj->bActive)
		{
			int powerupIndex = GetPowerup(pPowerupObj->objType);
			int playerIndex = GetPlayer(pPlayerObj->objType);

			// Clear any previous powerup the player had
			for (int i=0; i<icons.size(); i++)
			{
				if (icons[i]->playerAttached == playerIndex)
				{
					icons[i]->playerAttached = -1;
				}
			}

			// Assign the new powerup
			pPowerupObj->playerAttached = playerIndex;

			simulator.remove(*(pPowerupObj->rb));
			pPowerupObj->bActive = false;
		}

		// Handle Goals
		if (pGoalObj && pPigObj && pPigObj->bActive)
		{
			int goalIndex = GetGoal(pGoalObj->objType);

			playerScore[goalIndex]++;

			PlaySqueal(randRange(1,2), pPigObj->rb->x.x() / 3500.0f);
	
			if (goalIndex == lastPlayerScored)
			{
				playerStreak++;
			}
			else
			{
				playerStreak = 1;
				lastPlayerScored = goalIndex;
			}

			if (playerStreak > bestStreakVal)
			{
				bestStreakVal = playerStreak;
				bestStreakPlayer = goalIndex;
			}
			
			DetachFromPlayer(pPigObj);

			simulator.remove(*(pPigObj->rb));
			pPigObj->bActive = false;
		}
	}
}

const int kNumPigs = 100;
const int kNumShots = 20;

void LoadPhysObjects()
{
	// Load frame elements
	{
		int num_rb=0;
		int num_force=0;
		Rigid_Body **rb_list;
		Binary_Force **force_list;
		loadPhysicsFile("frame.physdb", 0, 0, 1.0f, 1.0f, &rb_list, &num_rb, &force_list, &num_force, &world_body);

		for (int i=0; i<num_rb; i++)
		{
			char buffer[1024];
			sprintf(buffer, "content/bitmaps/frame%2d.jpg", i);
		
			if ((i < 4) || (i >= 8)) frames.push_back(new game_object(buffer, rb_list[i], kFrame));
			if ((i >= 4) && (i < 8)) frames.push_back(new game_object(buffer, rb_list[i], (eObjType)(i - 4 + (int)kGoal1)));
		}
	}

	// Load pigs
	for (int i=0; i<kNumPigs; i++)
	{
		int num_rb=0;
		int num_force=0;
		Rigid_Body **rb_list;
		Binary_Force **force_list;
		float scale = 0.25f;
		loadPhysicsFile("pigs.physdb", 0, 0, scale, 0.01f, &rb_list, &num_rb, &force_list, &num_force, &world_body);

		rb_list[0]->x = Vec2(randRange(-200, 200), randRange(-200, 200));

		pigs.push_back(new game_object("content/bitmaps/pigs.jpg", rb_list[0], kPig));
	}

	// Load players
	for (int i=0; i<4; i++)
	{
		int num_rb=0;
		int num_force=0;
		Rigid_Body **rb_list;
		Binary_Force **force_list;
		loadPhysicsFile("player.physdb", 0, 0, 0.5f, 0.01f, &rb_list, &num_rb, &force_list, &num_force, &world_body);

		rb_list[0]->x = rb_list[1]->x = Vec2((i / 2 == 0) ? 400 : -400, (i % 2 == 0) ? 230 : -230);

		Handle h1(*(rb_list[0]), Vec2(0,8));
		Handle h2(*(rb_list[1]), Vec2(0,0)); 

		Pt2Pt_Constraint* pLocConstraint = new Pt2Pt_Constraint(h1, h2);
		simulator.add(*pLocConstraint);

		Angle_Spring_Force* pAngConstraint = new Angle_Spring_Force(h1, h2, 10000000, 50000);
		simulator.add(*pAngConstraint);

		char buffer[1024];
		sprintf(buffer, "content/bitmaps/player%2d.jpg", i);
		players.push_back(new game_object(buffer, rb_list[0], (eObjType)(i + (int)kPlayer1)));
		playerphantoms.push_back(new game_object("", rb_list[1], kPlayerPhantom));

		SetCollisionMask(rb_list[1], 1);
	}

	// Load shots
	for (int i=0; i<kNumShots; i++)
	{
		int num_rb=0;
		int num_force=0;
		Rigid_Body **rb_list;
		Binary_Force **force_list;
		loadPhysicsFile("shot.physdb", i * 40, 0, 0.5f, 0.5f, &rb_list, &num_rb, &force_list, &num_force, &world_body);

		rb_list[0]->x = Vec2(0,0);

		game_object* pGO = new game_object("content/bitmaps/shot.jpg", rb_list[0], kShot);
		pGO->bActive = false;
		simulator.remove(*(pGO->rb));
		shots.push_back(pGO);
	}

	// Load icons
	for (int i=0; i<5; i++)
	{
		int num_rb=0;
		int num_force=0;
		Rigid_Body **rb_list;
		Binary_Force **force_list;
		loadPhysicsFile("icon.physdb", i * 40, 0, 0.25f, 0.01f, &rb_list, &num_rb, &force_list, &num_force, &world_body);

		rb_list[0]->x = Vec2(0,0);

		game_object* pGO = 0;

		switch (i)
		{
			case 0: pGO = new game_object("content/bitmaps/icon.jpg", rb_list[0], kIconShot);  break;
			case 1: pGO = new game_object("content/bitmaps/icon.jpg", rb_list[0], kIconExplosion);  break;
			case 2: pGO = new game_object("content/bitmaps/icon.jpg", rb_list[0], kIconAttractor);  break;
			case 3: pGO = new game_object("content/bitmaps/icon.jpg", rb_list[0], kIconUp);  break;
			case 4: pGO = new game_object("content/bitmaps/icon.jpg", rb_list[0], kIconDown);  break;
		}
		pGO->bActive = false;
		simulator.remove(*(pGO->rb));
		icons.push_back(pGO);
	}
}

static int loopTractorSounds[4];

/******************************** Initialize ********************************/
/* the window and graphics subsystem are set up but still hidden            */
void Initialize( void )
{
	PS2Controllers.init();

	srand((unsigned)time(NULL));

    Globals.Gravity = 0.0f;

    Globals.MouseSpringKs = 128.0f;
    Globals.MouseSpringKd = 32.0f;
    Globals.MouseSpringRestLength = 0.0f;

	Globals.IsWinnerX = -70;
	Globals.IsWinnerY = 120;
	Globals.WinnerX = -30;
	Globals.WinnerY = 150;

	Globals.TopScoreX = -75;
	Globals.TopScoreY = 50;
	Globals.TopScoreValX = 60;
	Globals.TopScoreValY = 50;

	Globals.BestStreakX = -96;
	Globals.BestStreakY = 15;
	Globals.BestStreakValX = 60;
	Globals.BestStreakValY = 15;

	Globals.AnyKeyX = -155;
	Globals.AnyKeyY = -100;
    
    GetTime(); // throw away initial 0.0 so we're seconds-from-startup

    // gl stuff
    glEnable(GL_CULL_FACE);
    glEnable(GL_NORMALIZE);
    glEnable(GL_DEPTH_TEST);
    glShadeModel(GL_SMOOTH);

    // init our view
    Resize(WindowWidth,WindowHeight);

    // music init
    music.init();
    music.set_subdirectory("content/music/");
    music.setupnotes();
    music.loadseg("background", "simple_loop.sgt", MUSIC_TRACK_MAIN);

    // sound init
    zsoundInit(WindowHandle);

	InitializeFrameRate(kSimFramerate);

    // initialize the physics
    simulator.add(global_drag);
    simulator.add(gravity);
    simulator.add(world_body);

	LoadPhysObjects();

	for (int i=0; i<4; i++)
	{
		Handle playerHandle(*players[i]->rb, Vec2(0,0));
		Handle worldHandle(world_body, Vec2(0,0));

		player_thrust.push_back(new Constant_Force(playerHandle));
		simulator.add(*player_thrust[i]);

		player_torque.push_back(new Angle_Spring_Force(playerHandle, worldHandle, 500000.0f, 100000.0f));
		simulator.add(*player_torque[i]);
	}

	iconTextureShot = LoadTexture("content/bitmaps/iconShot.jpg");
	iconTextureExplosion = LoadTexture("content/bitmaps/iconExplosion.jpg");
	iconTextureAttractor = LoadTexture("content/bitmaps/iconAttractor.jpg");
	iconTextureUp = LoadTexture("content/bitmaps/iconUp.jpg");
	iconTextureDown = LoadTexture("content/bitmaps/iconDown.jpg");

	dozerTexture[0] = LoadTexture("content/bitmaps/dozer_red3.psd");
	dozerTexture[1] = LoadTexture("content/bitmaps/dozer_green.psd");
	dozerTexture[2] = LoadTexture("content/bitmaps/dozer_blue.psd");
	dozerTexture[3] = LoadTexture("content/bitmaps/dozer_yellow.psd");

	titleTexture = LoadTexture("content/bitmaps/title.jpg");

	backgroundTexture = LoadTexture("content/bitmaps/background.psd");
	pigTexture[0] = LoadTexture("content/bitmaps/pig1.psd");
	pigTexture[1] = LoadTexture("content/bitmaps/pig2.psd");
	pigTexture[2] = LoadTexture("content/bitmaps/pig3.psd");
	pigTexture[3] = LoadTexture("content/bitmaps/pig4.psd");
	pigTexture[4] = LoadTexture("content/bitmaps/pig5.psd");
	pigTexture[5] = LoadTexture("content/bitmaps/pig6.psd");
	pigTexture[6] = LoadTexture("content/bitmaps/pig7.psd");
	pigTexture[7] = LoadTexture("content/bitmaps/pig8.psd");
	pigTexture[8] = LoadTexture("content/bitmaps/pig9.psd");
	pigTexture[9] = LoadTexture("content/bitmaps/pig10.psd");
	pigTexture[10] = LoadTexture("content/bitmaps/pig11.psd");
	pigTexture[11] = LoadTexture("content/bitmaps/pig12.psd");
	pigTexture[12] = LoadTexture("content/bitmaps/pig13.psd");
	pigTexture[13] = LoadTexture("content/bitmaps/pig14.psd");
	pigTexture[14] = LoadTexture("content/bitmaps/pig15.psd");
	pigTexture[15] = LoadTexture("content/bitmaps/pig16.psd");
}

#define kPlayerSpeed  (15000.0f)

void CauseExplosion(const Vec2& loc, float dist, float power)
{
	for (int i=0; i<pigs.size(); i++)
	{
		if (pigs[i]->bActive)
		{
			float dist2 = (pigs[i]->rb->x - loc).len2();
			if (dist2 < (dist * dist))
			{
				dist2 = (dist * dist) - dist2;
				pigs[i]->rb->v += (pigs[i]->rb->x - loc).norm() * (dist2 * power / 1000.0f);
			}
		}
	}
}

void CauseAttraction(const Vec2& loc, float dist, float power)
{
	for (int i=0; i<pigs.size(); i++)
	{
		if (pigs[i]->bActive)
		{
			float dist2 = (pigs[i]->rb->x - loc).len2();
			if (dist2 < (dist * dist))
			{
				dist2 = (dist * dist) - dist2;
				pigs[i]->rb->v -= (pigs[i]->rb->x - loc).norm() * (dist2  * power / 1000.0f);
			}
		}
	}
}

static float GravDuration;
static float GravAmt;

void CauseUpGrav(float duration, float amt)
{
	GravDuration = duration;
	GravAmt = amt;
}

void CauseDownGrav(float duration, float amt)
{
	GravDuration = duration;
	GravAmt = -amt;
}

struct sScoreStruct
{
	int score;
	int player;
};


int scoreSortFunc( const void *arg1, const void *arg2 )
{
	sScoreStruct* pS1 = (sScoreStruct*)arg1;
	sScoreStruct* pS2 = (sScoreStruct*)arg2;

	if (pS1->score > pS2->score) return -1;
	if (pS1->score < pS2->score) return 1;
	return 0;
}

void SortScores(int pPlayerArray[4])
{
	sScoreStruct ss[4];

	for (int i=0; i<4; i++)
	{
		ss[i].score = playerScore[i];
		ss[i].player = i;
	}

	qsort(ss, 4, sizeof(sScoreStruct), scoreSortFunc);

	for (int i=0; i<4; i++)
	{
		pPlayerArray[i] = ss[i].player;
	}
}

void ResetGame()
{
	GameTimer = kGameDuration;
	fadeAlpha = 0.0f;

	GameDoneTimer = 0.0f;
	bGameDoneTimerSet = false;

	playerStreak = 0;
	lastPlayerScored = -1;

	bestStreakPlayer = 0;
	bestStreakVal = 0;

	for (int i=0; i<pigs.size(); i++)
	{

		pigs[i]->rb->x = Vec2(randRange(-200, 200), randRange(-200, 200));
		if (!pigs[i]->bActive)
		{
			simulator.add(*(pigs[i]->rb));
			pigs[i]->bActive = true;
		}
		pigs[i]->rb->update_derived_state();
	}

	for (int i=0; i<players.size(); i++)
	{
		players[i]->rb->x = playerphantoms[i]->rb->x = Vec2((i / 2 == 0) ? 400 : -400, (i % 2 == 0) ? 230 : -230);
		players[i]->rb->O = playerphantoms[i]->rb->O = 0;
		players[i]->rb->update_derived_state();
		playerScore[i] = 0;
	}

	for (int i=0; i<icons.size(); i++)
	{
		if (icons[i]->bActive)
		{
			simulator.remove(*(icons[i]->rb));
			icons[i]->bActive = false;
		}
		icons[i]->playerAttached = -1;
	}
}
		
void PlayGrunt(int which, float pan = 1.0f)
{
	char buffer[1024];
	sprintf(buffer, "content/sounds/grunt%d.wav", which);
	zsoundPlay(buffer, 0, (float)randRange(80, 100) / 100.0f, pan);
}

void PlaySqueal(int which, float pan)
{
	char buffer[1024];
	sprintf(buffer, "content/sounds/squeal%d.wav", which);
	zsoundPlay(buffer, 0, (float)randRange(80, 100) / 100.0f, pan);
}

#define kEndFadeDest  (0.9f)
#define kAnimDist     (10.0f)

static bool bGameHasStarted = false;
static int attractMode;
static float attractTime;

#define kAttractDelay   (1.0f)
#define kAttractActive  (12.0f)
#define kAttractShort   (6.0f)
#define kDelayUntilAttract (6.0f)

/*********************************** Idle ***********************************/
/* called by the message loop, and by paint messages                        */
void Idle( void )
{
	if (!bGameHasStarted)
	{
		PS2Controllers.sample();
		for (int i=0; i<PS2Controllers.num_joysticks(); i++)
		{
			if (PS2Controllers.button_went_down(Joystick::BUTTON_START, i))
			{
				ResetGame();
				bGameHasStarted = true;
				break;
			}
		}

		glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glDisable(GL_DEPTH_TEST);

		glLoadIdentity();
		glRotated(180.0*(-CameraAngleRadians/M_PI),0,0,1);
		glTranslatef(-CameraX,-CameraY,-CameraZ);

		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

		glColor3f(1,1,1);
		DrawStringOrthoPixels(-120, -250, "Press START to Begin");

		glColor3f(0.4f, 0.4f, 0.4f);
		DrawStringOrthoPixels(-450, -350, "Programming and Design by");
		DrawStringOrthoPixels(-370, -380, "Chris Carollo");

		DrawStringOrthoPixels(250, -350, "Art by");
		DrawStringOrthoPixels(130, -380, "Justin Hall and Chris Carollo");

		glEnable(GL_TEXTURE_2D);
		glColor3f(1,1,1);
		glBindTexture(GL_TEXTURE_2D, titleTexture);

		glBegin(GL_QUADS);
		glTexCoord2f(1, 1);
		glVertex2fv(Vec2(512, 350).m_v);
		glTexCoord2f(0, 1);
		glVertex2fv(Vec2(-512, 350).m_v);
		glTexCoord2f(0, 0);
		glVertex2fv(Vec2(-512, 94).m_v);
		glTexCoord2f(1, 0);
		glVertex2fv(Vec2(512, 94).m_v);
		glEnd();

		glDisable(GL_TEXTURE_2D);

		static Vec2 plowLoc[4];
		static Vec2 pigLoc[8];


		switch (attractMode)
		{
			case 0:  // startup
			{
				attractTime = GetTime();
				attractMode = 1;
				break;
			}

			case 1: // delay
			{
				if ((GetTime() - attractTime) > kAttractDelay)
				{
					// setup for two
					plowLoc[0] = Vec2(-800, -100);
					pigLoc[0] = Vec2(-700, -100);
					pigLoc[1] = Vec2(-650, -100);
					pigLoc[2] = Vec2(-600, -100);
					pigLoc[3] = Vec2(-550, -100);

					// proceed
					attractTime = GetTime();
					attractMode = 2;
				}
				break;
			}

			case 2: // run right
			{
				plowLoc[0] += Vec2(2, 0);
				for (int i=0; i<4; i++) pigLoc[i] += Vec2(2, 0);

				glEnable(GL_TEXTURE_2D);
				glColor3f(1,1,1);

				Vec2 xoff(36, 0);
				Vec2 yoff(0, 36);

				glBindTexture(GL_TEXTURE_2D, dozerTexture[3]);

				glBegin(GL_QUADS);
				glTexCoord2f(1, 0);
				glVertex2fv((plowLoc[0] + xoff + yoff).m_v);
				glTexCoord2f(1, 1);
				glVertex2fv((plowLoc[0] - xoff + yoff).m_v);
				glTexCoord2f(0, 1);
				glVertex2fv((plowLoc[0] - xoff - yoff).m_v);
				glTexCoord2f(0, 0);
				glVertex2fv((plowLoc[0] + xoff - yoff).m_v);
				glEnd();

				glBindTexture(GL_TEXTURE_2D, pigTexture[10]);
				xoff = Vec2(16, 0);
				yoff = Vec2(0, 16);

				for (int i=0; i<4; i++)
				{
					glBegin(GL_QUADS);
					glTexCoord2f(1, 0);
					glVertex2fv((pigLoc[i] + xoff + yoff).m_v);
					glTexCoord2f(0, 0);
					glVertex2fv((pigLoc[i] - xoff + yoff).m_v);
					glTexCoord2f(0, 1);
					glVertex2fv((pigLoc[i] - xoff - yoff).m_v);
					glTexCoord2f(1, 1);
					glVertex2fv((pigLoc[i] + xoff - yoff).m_v);
					glEnd();
				}

				glDisable(GL_TEXTURE_2D);

				// proceed
				if ((GetTime() - attractTime) > kAttractActive)
				{
					attractTime = GetTime();
					attractMode = 3;
				}
				break;
			}

			case 3: // delay
			{
				if ((GetTime() - attractTime) > kAttractDelay)
				{
					// setup for four
					plowLoc[0] = Vec2(550, -100);
					pigLoc[0] = Vec2(650, -100);
					pigLoc[1] = Vec2(700, -100);
					pigLoc[2] = Vec2(750, -100);
					pigLoc[3] = Vec2(800, -100);

					// proceed
					attractTime = GetTime();
					attractMode = 4;
				}
				break;
			}

			case 4: // run left
			{
				plowLoc[0] -= Vec2(2, 0);
				for (int i=0; i<4; i++) pigLoc[i] -= Vec2(2, 0);

				glEnable(GL_TEXTURE_2D);
				glColor3f(1,1,1);

				Vec2 xoff(36, 0);
				Vec2 yoff(0, 36);

				glBindTexture(GL_TEXTURE_2D, dozerTexture[3]);

				glBegin(GL_QUADS);
				glTexCoord2f(1, 1);
				glVertex2fv((plowLoc[0] + xoff - yoff).m_v);
				glTexCoord2f(0, 1);
				glVertex2fv((plowLoc[0] + xoff + yoff).m_v);
				glTexCoord2f(0, 0);
				glVertex2fv((plowLoc[0] - xoff + yoff).m_v);
				glTexCoord2f(1, 0);
				glVertex2fv((plowLoc[0] - xoff - yoff).m_v);
				glEnd();

				glBindTexture(GL_TEXTURE_2D, pigTexture[10]);
				xoff = Vec2(16, 0);
				yoff = Vec2(0, 16);

				for (int i=0; i<4; i++)
				{
					glBegin(GL_QUADS);
					glTexCoord2f(0, 0);
					glVertex2fv((pigLoc[i] + xoff + yoff).m_v);
					glTexCoord2f(1, 0);
					glVertex2fv((pigLoc[i] - xoff + yoff).m_v);
					glTexCoord2f(1, 1);
					glVertex2fv((pigLoc[i] - xoff - yoff).m_v);
					glTexCoord2f(0, 1);
					glVertex2fv((pigLoc[i] + xoff - yoff).m_v);
					glEnd();
				}

				glDisable(GL_TEXTURE_2D);

				// proceed
				if ((GetTime() - attractTime) > kAttractActive)
				{
					attractTime = GetTime();
					attractMode = 5;
				}
				break;
			}

			case 5:
			{
				float fade;
				
				if ((GetTime() - attractTime) < 1.0f)
				{
					fade = 0.0f;
				}
				if ((GetTime() - attractTime) < 2.0f)
				{
					fade = GetTime() - attractTime - 1.0f;
				}
				if ((GetTime() - attractTime) > (kAttractShort - 1.0f))
				{
					fade = 1.0f - ((GetTime() - attractTime) - (kAttractShort - 1.0f));
				}
				glColor3f(fade, fade, fade);

				glEnable(GL_TEXTURE_2D);
				glBindTexture(GL_TEXTURE_2D, iconTextureExplosion);

				Vec2 loc(-150, -87);
				Vec2 xoff(24, 0);
				Vec2 yoff(0, 24);

				glBegin(GL_QUADS);
				glTexCoord2f(0, 0);
				glVertex2fv((loc - xoff - yoff).m_v);
				glTexCoord2f(1, 0);
				glVertex2fv((loc + xoff - yoff).m_v);
				glTexCoord2f(1, 1);
				glVertex2fv((loc + xoff + yoff).m_v);
				glTexCoord2f(0, 1);
				glVertex2fv((loc - xoff + yoff).m_v);
				glEnd();

				glDisable(GL_TEXTURE_2D);

				glColor3f(fade, fade, 0);
				DrawStringOrthoPixels(-110, -100, "Use to cause an explosion");

				// proceed
				if ((GetTime() - attractTime) > kAttractShort)
				{
					attractTime = GetTime();
					attractMode = 6;
				}
				break;
			}

			case 6:
			{
				float fade;
				
				if ((GetTime() - attractTime) < 1.0f)
				{
					fade = 0.0f;
				}
				if ((GetTime() - attractTime) < 2.0f)
				{
					fade = GetTime() - attractTime - 1.0f;
				}
				if ((GetTime() - attractTime) > (kAttractShort - 1.0f))
				{
					fade = 1.0f - ((GetTime() - attractTime) - (kAttractShort - 1.0f));
				}
				glColor3f(fade, fade, fade);

				glEnable(GL_TEXTURE_2D);
				glBindTexture(GL_TEXTURE_2D, iconTextureAttractor);

				Vec2 loc(-120, -87);
				Vec2 xoff(24, 0);
				Vec2 yoff(0, 24);

				glBegin(GL_QUADS);
				glTexCoord2f(0, 0);
				glVertex2fv((loc - xoff - yoff).m_v);
				glTexCoord2f(1, 0);
				glVertex2fv((loc + xoff - yoff).m_v);
				glTexCoord2f(1, 1);
				glVertex2fv((loc + xoff + yoff).m_v);
				glTexCoord2f(0, 1);
				glVertex2fv((loc - xoff + yoff).m_v);
				glEnd();

				glDisable(GL_TEXTURE_2D);

				glColor3f(fade, fade, 0);
				DrawStringOrthoPixels(-80, -100, "Use to attract pigs");

				// proceed
				if ((GetTime() - attractTime) > kAttractShort)
				{
					attractTime = GetTime();
					attractMode = 7;
				}
				break;
			}

			case 7:
			{
				float fade;
				
				if ((GetTime() - attractTime) < 1.0f)
				{
					fade = 0.0f;
				}
				if ((GetTime() - attractTime) < 2.0f)
				{
					fade = GetTime() - attractTime - 1.0f;
				}
				if ((GetTime() - attractTime) > (kAttractShort - 1.0f))
				{
					fade = 1.0f - ((GetTime() - attractTime) - (kAttractShort - 1.0f));
				}
				glColor3f(fade, fade, fade);

				glEnable(GL_TEXTURE_2D);
				glBindTexture(GL_TEXTURE_2D, iconTextureUp);

				Vec2 loc(-180, -87);
				Vec2 xoff(24, 0);
				Vec2 yoff(0, 24);

				glBegin(GL_QUADS);
				glTexCoord2f(0, 0);
				glVertex2fv((loc - xoff - yoff).m_v);
				glTexCoord2f(1, 0);
				glVertex2fv((loc + xoff - yoff).m_v);
				glTexCoord2f(1, 1);
				glVertex2fv((loc + xoff + yoff).m_v);
				glTexCoord2f(0, 1);
				glVertex2fv((loc - xoff + yoff).m_v);
				glEnd();

				glDisable(GL_TEXTURE_2D);

				glColor3f(fade, fade, 0);
				DrawStringOrthoPixels(-140, -100, "Use to draw everything upward");
				// proceed
				if ((GetTime() - attractTime) > kAttractShort)
				{
					attractTime = GetTime();
					attractMode = 8;
				}
				break;
			}

			case 8:
			{
				float fade;
				
				if ((GetTime() - attractTime) < 1.0f)
				{
					fade = 0.0f;
				}
				if ((GetTime() - attractTime) < 2.0f)
				{
					fade = GetTime() - attractTime - 1.0f;
				}
				if ((GetTime() - attractTime) > (kAttractShort - 1.0f))
				{
					fade = 1.0f - ((GetTime() - attractTime) - (kAttractShort - 1.0f));
				}
				glColor3f(fade, fade, fade);

				glEnable(GL_TEXTURE_2D);
				glBindTexture(GL_TEXTURE_2D, iconTextureDown);

				Vec2 loc(-190, -87);
				Vec2 xoff(24, 0);
				Vec2 yoff(0, 24);

				glBegin(GL_QUADS);
				glTexCoord2f(0, 0);
				glVertex2fv((loc - xoff - yoff).m_v);
				glTexCoord2f(1, 0);
				glVertex2fv((loc + xoff - yoff).m_v);
				glTexCoord2f(1, 1);
				glVertex2fv((loc + xoff + yoff).m_v);
				glTexCoord2f(0, 1);
				glVertex2fv((loc - xoff + yoff).m_v);
				glEnd();

				glDisable(GL_TEXTURE_2D);

				glColor3f(fade, fade, 0);
				DrawStringOrthoPixels(-150, -100, "Use to draw everything downward");
				// proceed
				if ((GetTime() - attractTime) > kAttractShort)
				{
					attractTime = GetTime();
					attractMode = 9;
				}
				break;
			}

			case 9: // loop
			{
				attractMode = 0;
				break;
			}
		}

	    WaitFrameRate(GetFrameRate());
		SwapBuffers(WindowDC);

		if (!bGameHasStarted) return;
	}

	if (GravDuration > 0.0f)
	{
		GravDuration -= 1.0f / GetFrameRate();
		gravity.m_g.set_y(GravAmt);
	}
	else
	{
		gravity.m_g.set_y(0.0f);
	}

	GameTimer -= 1.0f / GetFrameRate();
	if (GameTimer < 0.0f)
	{
		if (fadeAlpha < kEndFadeDest)
		{
			fadeAlpha += 2.0f / GetFrameRate();
		}
	}
    
	PS2Controllers.sample();

	/**************************** update sounds ****************************/   

	for (int i=0; i<4; i++)
	{
		if (loopTractorSounds[i] = 0)
			loopTractorSounds[i] = zsoundPlay("content/sounds/tractor2.wav", 1, 0.5, players[i]->rb->x.x() / 3500.0f);

		zsoundChangePan(loopTractorSounds[i], players[i]->rb->x.x() / 3500.0f);
		zsoundChangeVolume(loopTractorSounds[i], min(0.5 + (players[i]->rb->v.len() / 500.0f), 0.8f));
	}

	/**************************** step the simulator ****************************/   

    if (spring)                             //update
        spring->m_a = Handle(world_body, world_mouse);

	int const NumberOfSimSteps = 1;
    for(int StepCounter = 0;StepCounter < NumberOfSimSteps;++StepCounter) {
        // step as many times as you like
        simulator.step(1/(NumberOfSimSteps*GetFrameRate()));
		handleCollisions();
        // do any game stuff you want here (like per-step collision response, etc.)
    }
    // now wait for the rest of the time so we're consistent on fast machines
    WaitFrameRate(GetFrameRate());

	// Potentially spawn icons
	for (int i=1; i<icons.size(); i++)
	{
		if (!icons[i]->bActive && (icons[i]->playerAttached == -1) && (randRange(0, iconSpawnSeconds[i] * kSimFramerate) == 7))
		{
			icons[i]->rb->x = Vec2(randRange(-400, 400), randRange(-300, 300));
			icons[i]->bActive = true;
			simulator.add(*(icons[i]->rb));
		}
	}

	/**************************** handle joystick input *****************************/

	if (GameTimer > 0.0f)
	{
		for (int i=0; i<PS2Controllers.num_joysticks(); i++)
		{
			// Thrust/rotation
			float rotation = PS2Controllers.right_stick(i).x();
			float currot = -players[i]->rb->O.z();
			float speed = players[i]->rb->v.len();

			Vec2 dir(cos(currot - 1.570796326794895f), -sin(currot - 1.570796326794895f));

			// Explicit deceleration when not thrusting enough
			float thrust = PS2Controllers.left_stick(i).y() * kPlayerSpeed;
			if (thrust > (kPlayerSpeed / 10) || (thrust < -kPlayerSpeed / 10)) 
			{
				player_thrust[i]->m_f.set_x(thrust * dir.x());
				player_thrust[i]->m_f.set_y(thrust * dir.y());
			}
			else
			{
				player_thrust[i]->m_f.set_x(-players[i]->rb->v.x() * 100.0f);
				player_thrust[i]->m_f.set_y(-players[i]->rb->v.y() * 100.0f);
			}

			if ((dir.x() * players[i]->rb->v.x()) + (dir.y() * players[i]->rb->v.y()) > 0)
			{
				players[i]->rb->v = dir * speed;
			}
			else
			{
				players[i]->rb->v = dir * -speed;
			}

			// Turn left or right
			player_torque[i]->m_rest_len = currot + rotation;

			if (PS2Controllers.button_went_down(Joystick::BUTTON_RIGHT_D, i))
			{
				for (int j=0; j<icons.size(); j++)
				{
					if (icons[j]->playerAttached == i)
					{
						switch (j)
						{
							case 1: CauseExplosion(players[i]->rb->x, 300, 10);  break;
							case 2: CauseAttraction(players[i]->rb->x, 600, 2);  break;
							case 3: CauseUpGrav(2, 500);  break;
							case 4: CauseDownGrav(2, 500);  break;
						}
						icons[j]->playerAttached = -1;
					}
				}
			}
		}
	}
	else
	{
		for (int i=0; i<4; i++)
		{
			if (PS2Controllers.button_went_down(Joystick::BUTTON_RIGHT_U, i))
			{
				ResetGame();
				break;
			}
		}

		for (int i=0; i<PS2Controllers.num_joysticks(); i++)
		{
			player_thrust[i]->m_f.set_x(0);
			player_thrust[i]->m_f.set_y(0);
		}
	}

	/**************************** render everything *****************************/
   
    // lots of gl stuff
	glClearColor(1.0f, 0.0f, 0.0f, 0.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glDisable(GL_DEPTH_TEST);


    glLoadIdentity();
    glRotated(180.0*(-CameraAngleRadians/M_PI),0,0,1);
    glTranslatef(-CameraX,-CameraY,-CameraZ);

    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	static float colormap[4][3] = { { 1.0f, 0.0f, 0.0f }, { 0.0f, 1.0f, 0.0f }, { 0.0f, 0.0f, 1.0f }, { 1.0f, 1.0f, 0.0f } };

	// Render the background
	glEnable(GL_TEXTURE_2D);
	glColor3f(1,1,1);
	glBindTexture(GL_TEXTURE_2D, backgroundTexture);

	glBegin(GL_QUADS);
	glTexCoord2f(1, 0);
	glVertex2fv(Vec2(WindowWidth/2 + 12, WindowHeight/2).m_v);
	glTexCoord2f(0, 0);
	glVertex2fv(Vec2(-WindowWidth/2 - 12, WindowHeight/2).m_v);
	glTexCoord2f(0, 1);
	glVertex2fv(Vec2(-WindowWidth/2 - 12, -WindowHeight/2).m_v);
	glTexCoord2f(1, 1);
	glVertex2fv(Vec2(WindowWidth/2 + 12, -WindowHeight/2).m_v);
	glEnd();

	glDisable(GL_TEXTURE_2D);

#if 0
    glColor3f(0,0,0);
    for (int i=0; i<4; i++)
	{
		assert(frames[i]);
		game_object &obj = *frames[i];

		glBody1v(obj.rb->m_Pbody.get(), true);
	}

	// Draw frame and goals
	for (int i=4; i<8; i++)
	{
		assert(frames[i]);
		game_object &obj = *frames[i];

		glColor3f(colormap[i-4][0], colormap[i-4][1], colormap[i-4][2]);
		glBody1v(obj.rb->m_Pbody.get(), true);
	}

#endif

	// Draw pigs
	glEnable(GL_TEXTURE_2D);
	glColor3f(1,1,1);
	for (int i=0; i<pigs.size(); i++)
	{
		game_object &obj = *pigs[i];
		if (obj.bActive)
		{
			Vec2 loc = obj.rb->x;
			Vec2 xoff(16, 0);
			Vec2 yoff(0, 16);

			glBindTexture(GL_TEXTURE_2D, pigTexture[obj.animFrame]);

			glBegin(GL_QUADS);
			glTexCoord2f(0, 0);
			glVertex2fv((loc - xoff - yoff).m_v);
			glTexCoord2f(1, 0);
			glVertex2fv((loc + xoff - yoff).m_v);
			glTexCoord2f(1, 1);
			glVertex2fv((loc + xoff + yoff).m_v);
			glTexCoord2f(0, 1);
			glVertex2fv((loc - xoff + yoff).m_v);
			glEnd();

			if ((pigs[i]->lastAnimLoc - pigs[i]->rb->x).len2() > (kAnimDist * kAnimDist))
			{
				obj.lastAnimLoc = pigs[i]->rb->x;
				obj.animFrame = (obj.animFrame + 1) % 16;

				if (obj.animFrame == 0)
				{
					PlayGrunt(randRange(1, 8), pigs[i]->rb->x.x() / 3500.0f);
				}
			}
		}
	}
	glDisable(GL_TEXTURE_2D);

	// Draw player ships
    glEnable(GL_TEXTURE_2D);
	for (int i=0; i<players.size(); i++)
	{
		game_object &obj = *players[i];

#if 0
		glDisable(GL_TEXTURE_2D);
		glColor3f(colormap[i][0], colormap[i][1], colormap[i][2]);
		glBody1v(playerphantoms[i]->rb->m_Pbody.get(), true);
	    glEnable(GL_TEXTURE_2D);
#endif
		float currot = -players[i]->rb->O.z();
		Vec2 dir(cos(currot - 1.570796326794895f), -sin(currot - 1.570796326794895f));
		Vec2 perp = dir.perp();

		glColor3f(1,1,1);
		glBindTexture(GL_TEXTURE_2D, dozerTexture[i]);

		Vec2 loc = players[i]->rb->x;

		dir *= -36;
		perp *= 36;

		glBegin(GL_QUADS);
        glTexCoord2f(0, 0);
        glVertex2fv((loc - perp - dir).m_v);
        glTexCoord2f(1, 0);
        glVertex2fv((loc + perp - dir).m_v);
        glTexCoord2f(1, 1);
        glVertex2fv((loc + perp + dir).m_v);
        glTexCoord2f(0, 1);
		glVertex2fv((loc - perp + dir).m_v);
        glEnd();

	}
	glDisable(GL_TEXTURE_2D);

	// Draw shots
	glColor3f(1,1,0);
	for (int i=0; i<shots.size(); i++)
	{
		game_object &obj = *shots[i];
		if (obj.bActive)
		{
			glBody1v(obj.rb->m_Pbody.get(), true);
		}
	}

	// Draw player scores
	for (int i=0; i<4; i++)
	{
		char buffer[1024];
		sprintf(buffer, "%02d", playerScore[i]);
		glColor3f(colormap[i][0], colormap[i][1], colormap[i][2]);

		float x = (float)(-307 + (208 * i)) / 1024 * WindowWidth;
		float y = (float)(356) / 768 * WindowHeight; 
		DrawStringOrthoPixels(x, y, buffer);
	}

	// Draw timer
	if (GameTimer > 0.0f)
	{
		char buffer[1024];
		sprintf(buffer, "%02.0f", GameTimer);
		glColor3f(1,1,1);
		DrawStringOrthoPixels(0, -384, buffer);
	}

	// Draw icons
    glEnable(GL_TEXTURE_2D);

    glColor3f(1,1,1);
	for (int i=0; i<icons.size(); i++)
	{
		if (icons[i]->bActive || (icons[i]->playerAttached >= 0))
		{
			switch (i)
			{
				case 0: glBindTexture(GL_TEXTURE_2D, iconTextureShot);  break;
				case 1: glBindTexture(GL_TEXTURE_2D, iconTextureExplosion);  break;
				case 2: glBindTexture(GL_TEXTURE_2D, iconTextureAttractor);  break;
				case 3: glBindTexture(GL_TEXTURE_2D, iconTextureUp);  break;
				case 4: glBindTexture(GL_TEXTURE_2D, iconTextureDown);  break;
			}
		
			Vec2f loc, xoff, yoff;

			if (icons[i]->bActive)
			{
				xoff = Vec2(24, 0);
				yoff = Vec2(0, 24);
				loc = icons[i]->rb->x;
			}
			else
			{
				xoff = Vec2(13, 0);
				yoff = Vec2(0, 13);

				loc = Vec2(-307 + (208 * icons[i]->playerAttached) - 20, 369);
			}

			glBegin(GL_QUADS);
            glTexCoord2f(0, 0);
            glVertex2fv((loc - xoff - yoff).m_v);
            glTexCoord2f(1, 0);
            glVertex2fv((loc + xoff - yoff).m_v);
            glTexCoord2f(1, 1);
            glVertex2fv((loc + xoff + yoff).m_v);
            glTexCoord2f(0, 1);
			glVertex2fv((loc - xoff + yoff).m_v);
            glEnd();
		}
	}

	glDisable(GL_TEXTURE_2D);

	// Game results
	if ((GameTimer < 0.0f) && (fadeAlpha >= kEndFadeDest))
	{
		if ((GetTime() - GameDoneTimer) > kDelayUntilAttract)
		{
			bGameDoneTimerSet = false;
			bGameHasStarted = false;
			attractMode = 0;
		}

		glColor4f(0,0,0, fadeAlpha);

		glBegin(GL_QUADS);
        glVertex2fv(Vec2(-WindowWidth/2 - 12, -WindowHeight/2).m_v);
        glVertex2fv(Vec2(WindowWidth/2 + 12, -WindowHeight/2).m_v);
        glVertex2fv(Vec2(WindowWidth/2 + 12, WindowHeight/2).m_v);
		glVertex2fv(Vec2(-WindowWidth/2 - 12, WindowHeight/2).m_v);
        glEnd();
		static int endSound = zsoundPlay("content/sounds/winsound.wav", 0, 1.0f);

		char buffer[1024];

		int playerRank[4];
		SortScores(playerRank);

		char nameMap[4][10] = { "Red", "Green", "Blue", "Yellow" };

		int winner = playerRank[0];
		glColor3f(colormap[winner][0], colormap[winner][1], colormap[winner][2]);
		DrawStringOrthoPixels(Globals.WinnerX, Globals.WinnerY, nameMap[winner]);
		glColor3f(1,1,1);
		DrawStringOrthoPixels(Globals.IsWinnerX, Globals.IsWinnerY, "player wins!");

		for (int i=0; i<4; i++)
		{
			glColor3f(colormap[playerRank[i]][0], colormap[playerRank[i]][1], colormap[playerRank[i]][2]);
			DrawStringOrthoPixels(-75, 50 - (i * 24), nameMap[playerRank[i]]);

			sprintf(buffer, "%02d", playerScore[playerRank[i]]);
			glColor3f(1,1,1);
			DrawStringOrthoPixels(30, 50 - (i * 24), buffer);
		}

#if 0
		glColor3f(1,1,1);
		DrawStringOrthoPixels(Globals.BestStreakX, Globals.BestStreakY, "Best Streak:");

		sprintf(buffer, "%02d", bestStreakVal);
		glColor3f(colormap[bestStreakPlayer][0], colormap[bestStreakPlayer][1], colormap[bestStreakPlayer][2]);
		DrawStringOrthoPixels(Globals.BestStreakValX, Globals.BestStreakValY, buffer);
#endif

		glColor3f(1,1,1);
		DrawStringOrthoPixels(Globals.AnyKeyX, Globals.AnyKeyY, "Press Triangle to Play Again");
	}
	else if (GameTimer < 0.0f) // fade down
	{
		if (!bGameDoneTimerSet)
		{
			GameDoneTimer = GetTime();
			bGameDoneTimerSet = true;
		}

		glColor4f(0,0,0, fadeAlpha);

		glBegin(GL_QUADS);
        glVertex2fv(Vec2(-WindowWidth/2 - 12, -WindowHeight/2).m_v);
        glVertex2fv(Vec2(WindowWidth/2 + 12, -WindowHeight/2).m_v);
        glVertex2fv(Vec2(WindowWidth/2 + 12, WindowHeight/2).m_v);
		glVertex2fv(Vec2(-WindowWidth/2 - 12, WindowHeight/2).m_v);
        glEnd();
	}



    //DrawTimingBars();

    SwapBuffers(WindowDC);
}

/******************************** AppWndProc ********************************/
/* lame to expose this, but the alternatives all suck too                   */
LONG FAR PASCAL AppWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
    switch (msg)
    {
        // keyboard
        case WM_CHAR:
            break;

        case WM_LBUTTONDOWN:
        {
			SetCapture(hwnd);

            Vec2 body_space_position;
            Rigid_Body* rb = simulator.pick(body_space_position, world_mouse);
            if (rb)
            {
                spring = new Spring_Force(
                    Handle(world_body, world_mouse),
                    Handle(*rb, body_space_position), 
                    Globals.MouseSpringKs,
                    Globals.MouseSpringKd,
                    Globals.MouseSpringRestLength
                    );
                simulator.add(*spring);
            }
        } break;

        case WM_LBUTTONUP:
        {
			if(!(wParam & MK_MBUTTON & MK_RBUTTON))
			{
				ReleaseCapture();
			}

            ClearMouseSpring();
        } break;

		case WM_KEYDOWN: {
            switch(wParam) {
                case VK_ESCAPE: {
                    AppQuit = true;
                    break;
                }
            }
            break;
        }
        case WM_SYSKEYDOWN:
        case WM_SYSKEYUP:
        case WM_KEYUP:
            break;

            // mouse
        case WM_MBUTTONDOWN:
        case WM_MBUTTONUP:
            break;

        case WM_MOUSEMOVE: {
			Vec2 new_screen_mouse = Vec2(float(GET_X_LPARAM(lParam)),float(GET_Y_LPARAM(lParam)));
            screen_mouse = new_screen_mouse;
            world_mouse = WorldFromWindowCoords(screen_mouse);
            break;
        }

         
            /**************** below is stuff you probably don't care about ****************/
         
        case WM_DESTROY: {
            PostQuitMessage(0);
            break;
        }
        case WM_PAINT: {
            PAINTSTRUCT ps;
            BeginPaint(hwnd,&ps);
            Idle();
            EndPaint(hwnd,&ps);
            return 0L;
        }
        case WM_SIZE: {
            Resize(LOWORD(lParam),HIWORD(lParam));
            break;
        }
    }
    return (LONG)DefWindowProc(hwnd,msg,(LONG)wParam,(LONG)lParam);
}

/********************************** Resize **********************************/
/* called on WM_SIZE, and at the bottom of initialize                       */
void Resize( int w, int h )
{
    // compute our frustrum planes, the x,y plane values are on the near clip plane
    WindowWidth = w;
    WindowHeight = h;
    AspectRatio = float(w)/float(h);
    float ymax = ZNear * float(tan(FOVY/2.0));
	ymax = (float)h / 13800;
    float ymin = -ymax;
    float xmin = AspectRatio * ymin;
    float xmax = AspectRatio * ymax;
   
    // load the viewport and projection matrix, gl stuff
    glViewport(0,0,w,h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    // gluPerspective doesn't set the near plane as the projection
    // plane, so you can't do good math, so do it ourselves
    glFrustum(xmin,xmax,ymin,ymax,ZNear,ZFar);
    glMatrixMode(GL_MODELVIEW);
}

/******************************* Uninitialize *******************************/
/* called before the graphics is destroyed, but the window is already toast */
void Uninitialize( void )
{
}

/********************************* WinMain **********************************/

int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrev, LPSTR szCmdLine, int sw)
{
    // register and create our window
    WNDCLASS cls;
    if(!hPrev) {
        cls.hCursor        = LoadCursor(NULL,IDC_ARROW);
        cls.hIcon          = 0;
        cls.lpszMenuName   = 0;
        cls.lpszClassName  = AppName;
        cls.hbrBackground  = 0;
        cls.hInstance      = hInst;
        cls.style          = CS_VREDRAW | CS_HREDRAW | CS_DBLCLKS | CS_OWNDC;
        cls.lpfnWndProc    = (WNDPROC)AppWndProc;
        cls.cbWndExtra     = 0;
        cls.cbClsExtra     = 0;

        if (!RegisterClass(&cls))
            return FALSE;
    }

    DWORD Style = WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
    RECT WindowRect = { 0, 0 };
    WindowRect.right = WindowWidth;
    WindowRect.bottom = WindowHeight;
    BOOL Menu = FALSE;

    AdjustWindowRect(&WindowRect,Style,Menu);
    
    WindowHandle = CreateWindow(AppName,AppName,Style,
                                0,50,
                                WindowRect.right-WindowRect.left,
                                WindowRect.bottom-WindowRect.top,
                                0,0,hInst,0);

    // create our gl stuff
    WindowDC = GetDC(WindowHandle);
    PIXELFORMATDESCRIPTOR FormatDescriptor = {
        sizeof (PIXELFORMATDESCRIPTOR), 1,
        PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER,
        PFD_TYPE_RGBA, 32,   // 32bpp
        0,0, 0,0, 0,0, 0,0,
        0, 0,0,0,0,
        24, 8,               // depth, stencil
        0,
        PFD_MAIN_PLANE,
        0,
        0,0,0
    };
    int FormatIndex = ChoosePixelFormat(WindowDC,&FormatDescriptor);
    SetPixelFormat(WindowDC,FormatIndex,&FormatDescriptor);
    WindowRC = wglCreateContext(WindowDC);
    wglMakeCurrent(WindowDC,WindowRC);

    // okay, ready to go
   
    Initialize();    // call the app
   
    ShowWindow(WindowHandle,sw);
   
    // pump the message queue
    MSG msg;
    while(!AppQuit) {
        if(PeekMessage(&msg,0,0,0,PM_REMOVE)) {
            if(msg.message == WM_QUIT)
                break;
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        } else {
            Idle();    // call the app
        }
    }

    Uninitialize();  // call the app
   
    // clean up gl stuff and exit
    wglMakeCurrent(0,0);
    wglDeleteContext(WindowRC);
    ReleaseDC(WindowHandle,WindowDC);
   
    return int(msg.wParam);
}

/********************** utilities that use our globals **********************/
Vec2 WorldFromWindowCoords( Vec2 screen )
{
    float xn = float(screen.x())/WindowWidth - 0.5f;
    float yn = float(WindowHeight-screen.y())/WindowHeight - 0.5f;
    float xc = float(xn*2.0*CameraZ*tan(FOVY/2)*AspectRatio);
    float yc = float(yn*2.0*CameraZ*tan(FOVY/2));
    return Vec2(CameraX,CameraY) + Matrix22::rotation(CameraAngleRadians) * Vec2(xc,yc);
}
Vec2 WindowFromWorldCoords( Vec2 world )
{
    Vec2 c = Matrix22::rotation(-CameraAngleRadians) * (world - Vec2(CameraX,CameraY));
    float xn = float(c.x()/(2.0*CameraZ*tan(FOVY/2)*AspectRatio));
    float yn = float(c.y()/(2.0*CameraZ*tan(FOVY/2)));
    float xs = WindowWidth*(xn + 0.5f);
    float ys = WindowHeight - WindowHeight*(yn + 0.5f);
    return Vec2(xs,ys);
}
