/*
 * glutils.cpp - see utils.h for "details"
 * checker@d6.com
 *
 */

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <math.h>
#include <gl/gl.h>
#include <gl/glu.h>
#include <assert.h>
#include "glutils.h"
#include "utils.h"

/******************************** font stuff *********************************/

int unsigned Base = 0;
int unsigned TextWidth, TextHeight;
int unsigned TextDescenderHeight;

int unsigned GetFontBase( void )
{
   if(Base) {
      return Base;
   } else {
      Base = glGenLists(256);  assert(Base);
      HDC DC = GetDC(0);
      HFONT OldFont = SelectFont(DC,
                                 CreateFont(28,
                                            0,0,0,
                                            FW_BOLD,FALSE,FALSE,FALSE,ANSI_CHARSET,
                                            OUT_DEFAULT_PRECIS,
                                            CLIP_DEFAULT_PRECIS,
                                            DEFAULT_QUALITY,
                                            DEFAULT_PITCH,
                                            "Tahoma"));
      TEXTMETRIC tm;
      GetTextMetrics(DC,&tm);
      TextWidth = tm.tmMaxCharWidth;
      TextHeight = tm.tmHeight;
      TextDescenderHeight = tm.tmDescent;
      if(!wglUseFontBitmaps(DC, 0, 255, Base)) {
         return 0;
      }
      DeleteFont(SelectFont(DC,OldFont));
      ReleaseDC(0,DC);
      return Base;
   }
}

void DrawStringOrthoPixels( float x, float y, char *s )
{
   glListBase(GetFontBase());
   glRasterPos2f(x,y+TextDescenderHeight+2);
   glCallLists((GLsizei)strlen(s),GL_UNSIGNED_BYTE,s);
}
void DrawString( float x, float y, char *s)
{
   PushOrthoPixels();
   DrawStringOrthoPixels(x,y,s);
   PopOrthoPixels();
}
void PushOrthoPixels( void )
{
   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   int vp[4];
   glGetIntegerv(GL_VIEWPORT,vp);
   glOrtho(0,vp[2],0,vp[3],-1,1);
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glLoadIdentity();
}
void PopOrthoPixels( void )
{
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW);
}

float GetTextHeight( void )
{
   return float(TextHeight);
}
float GetTextWidth( char *s )
{
   return float(TextWidth * strlen(s));
}

/*************************** DrawTimingBars stuff ****************************/

static void DrawSegmentedBar( float x, float y, float xbar, float ybar,
                              float TotalLength, float SegmentLength,
                              float aaColors[2][3], int ShadeCount = 0 )
{
   float TotalDivSegment = TotalLength / SegmentLength;
   float TotalModSegmentScale = float(fmod(TotalLength,SegmentLength)) / SegmentLength;
   int unsigned NumberOfSegments = (int unsigned)TotalDivSegment;
   int unsigned ColorIndex = 0;
   float z = -0.2f;

   for(int unsigned i = 0;i < NumberOfSegments;i++)
   {
      glBegin(GL_QUADS);
      float s = 1.f;
      if(ShadeCount && ((i / ShadeCount) & 1))
      {
         s = 0.5f;
      }
      glColor3f(s*aaColors[ColorIndex][0],
                s*aaColors[ColorIndex][1],
                s*aaColors[ColorIndex][2]);
      glVertex3f(x     ,y     ,z);
      glVertex3f(x+xbar,y     ,z);
      glVertex3f(x+xbar,y+ybar,z);
      glVertex3f(x     ,y+ybar,z);
      glEnd();
      ColorIndex = !ColorIndex;
      x += xbar;
   }

   xbar *= TotalModSegmentScale;
   glBegin(GL_QUADS);
   float s = 1.f;
   if(ShadeCount && ((i / ShadeCount) & 1))
   {
      s = 0.5f;
   }
   glColor3f(s*aaColors[ColorIndex][0],
             s*aaColors[ColorIndex][1],
             s*aaColors[ColorIndex][2]);
   glVertex3f(x     ,y     ,z);
   glVertex3f(x+xbar,y     ,z);
   glVertex3f(x+xbar,y+ybar,z);
   glVertex3f(x     ,y+ybar,z);
   glEnd();
}

void DrawTimingBars( void )
{
   static double LastTime = GetTime()-0.01;                 // first call's data is garbage, set to at least 100fps
   static double CurrentTime;

   int const NUM_SAMPLES = 64;
   static double RecentFrames[NUM_SAMPLES], FrameTotal=0.0; // secretly im counting on the array being Zeroed as well
   static int    FrameIdx=0;                                // where in the Frame sample storage array we are
   static int    ValidData=0;                               // is the data valid (i.e. have we looped through it all once?)

   CurrentTime = GetTime();
   double FPS = 1.0/(CurrentTime - LastTime);
   LastTime = CurrentTime;

   FrameTotal-=RecentFrames[FrameIdx];
   RecentFrames[FrameIdx]=FPS;
   FrameTotal+=FPS;
   FrameIdx=(FrameIdx+1)%NUM_SAMPLES;
   if (FrameIdx==0) ValidData=1; // once we have wrapped once, we have valid hyst data

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glOrtho(0.f,1.f,0.f,1.f,0.f,10.f);
   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glLoadIdentity();
   // @todo need to glPushAttrib and ensure the correct state
   glDisable(GL_DEPTH_TEST);

   float const border = 0.01f;
   float const width = 1.0f - 2.0f * border;
   float const height = 0.0125f;
   float const d_background = 0.005f;
   float const FPSAcross = 100.f;

   float w = float(FPS) / FPSAcross;
   float s = (float)ceil(w);
   float e = width * w;
   float t = width * (float(FrameTotal/NUM_SAMPLES) / FPSAcross);
   float back_seg_length = width / 4.f;
   float front_seg_length = back_seg_length / 5.f;

   static float s_current = s;
   static float s_desired = s;
   static double s_time;
   if(s >= s_current)
   {
      // we need to rescale FPS bar immediately
      s_current = s;
      s_time = GetTime();
   }
   else if(s < s_current)
   {
      // we'd like to rescale, is it time?
      if(s_desired >= s)
      {
         if((GetTime() - s_time) > 2.0f)
         {
            // we've been stable for a bit
            s_current = s;
         }
      }
      else
      {
         s_time = GetTime();
         s_desired = s;
      }
   }
    
   e /= s_current;
   t /= s_current;
   back_seg_length /= s_current;
   front_seg_length /= s_current;

   float x = border, y = border, xbar = front_seg_length, ybar = height;
   float aaColors[2][2][3] =
      {
         { { 1.0f, 0.0f, 0.0f }, { 0.0f, 1.0f, 0.0f } },
         { { 1.0f, 1.0f, 1.0f }, { 0.0f, 0.0f, 1.0f } }
      };

   if(0)
   {
      float x = 0, y = 0;
      float xxbar = xbar * 5.f, yybar = ybar+2*d_background;
      glBegin(GL_QUADS);
      glColor3f(0,0,0);

      glVertex3f(x,y,0);
      glVertex3f(x+xxbar,y,0);
      glVertex3f(x+xxbar,y+yybar,0);
      glVertex3f(x,y+yybar,0);

      glVertex3f(0,+.5,0);
      glVertex3f(0+.1f,+.5f,0);
      glVertex3f(0+.1f,+.5f+.1f,0);
      glVertex3f(0,+.5f+.1f,0);
            
      glEnd();
   }
    
   DrawSegmentedBar(x,y-d_background,
                    xbar*5.f,ybar+2.f*d_background,
                    e,back_seg_length,
                    aaColors[0],4);

   if(ValidData)
   {
      DrawSegmentedBar(x,y,
                       xbar,ybar,
                       t,front_seg_length,
                       aaColors[1],20);
   }

   glEnable(GL_DEPTH_TEST);
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW);
}

/**************************** misc drawing utils *****************************/

void DrawAxes( float scale, float x, float y, float z )
{
   glPushMatrix();
   glTranslatef(x,y,z);
   glScalef(scale,scale,scale);

   glBegin(GL_LINES);

   // x
   glColor3f(1.0,0.0,0.0);
   glVertex3f(0.0,0.0,0.0);
   glVertex3f(1.0,0.0,0.0);
   // letter 'x'
   glVertex3f(0.8f,0.05f,0.0);  glVertex3f(1.0,0.25f,0.0);
   glVertex3f(0.8f,0.25f,0.0);  glVertex3f(1.0,0.05f,0.0);

   // y
   glColor3f(0.0,1.0,0.0);
   glVertex3f(0.0,0.0,0.0);
   glVertex3f(0.0,1.0,0.0);

   // z
   glColor3f(0.0,0.0,1.0);
   glVertex3f(0.0,0.0,0.0);
   glVertex3f(0.0,0.0,1.0);

   glEnd();

   glPopMatrix();
}

/****************************** jon's texture *******************************/

GLuint gl_texture_from_bitmap(unsigned char *bits, int width, int height,
                              int bytes_per_pixel) {
    GLuint texture_id;
    glGenTextures(1, &texture_id);

    glBindTexture(GL_TEXTURE_2D, texture_id);

    int format;
    if (bytes_per_pixel == 3) {
        format = GL_RGB;
    } else {
        format = GL_RGBA;
    }

    gluBuild2DMipmaps(GL_TEXTURE_2D, bytes_per_pixel, width, height, format,
                      GL_UNSIGNED_BYTE, (void *)bits);

    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,
                    GL_LINEAR_MIPMAP_LINEAR);

    glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

    return texture_id;
}
